      SUBROUTINE EMFACT (QARG)
C***********************************************************************
C                 EMFACT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Applies Variable Emission Rate and
C                 Unit Conversion Factors
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C        MODIFIED  : for handling OpenPit Source Type - PES, 7/26/94
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To include an option to vary emissions by season,
C                    hour-of-day, and day-of-week (SHRDOW).
C                    R.W. Brode, PES, 4/10/2000
C
C        INPUTS:  Arrays of Source Parameters
C                 Date and Hour
C                 Meteorological Variables for One Hour
C                 Variable Emission Rate Flags and Factors
C                 Unit Conversion Rate Factors
C
C        OUTPUTS: Adjusted Emission Rate, QTK
C
C        CALLED FROM:   PCALC
C                       VCALC
C                       ACALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: QARG

C     Variable Initializations
      MODNAM = 'EMFACT'

C     Apply Emission Unit Factor (EMIFAC) and Variable Emission Rate
C     Factor, Based on Value of QFLAG
      IF (QFLAG(ISRC) .EQ. ' ') THEN
         QTK = QARG

C*----   ISCSTM Modification: To handle hourly emissions - jah 11/4/94
      ELSE IF (QFLAG(ISRC) .EQ. 'HOURLY') THEN
         QTK = QARG
C*----
C*#

      ELSE IF (QFLAG(ISRC) .EQ. 'MONTH') THEN
         QTK = QARG * QFACT(IMONTH,ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'HROFDY') THEN
         QTK = QARG * QFACT(IHOUR,ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'STAR') THEN
         QTK = QARG * QFACT((IUCAT+(KST-1)*NWSCAT),ISRC)
      ELSE IF (QFLAG(ISRC) .EQ. 'SEASON') THEN
         QTK = QARG * QFACT(ISEAS,ISRC)
      ELSE IF (QFLAG(ISRC). EQ. 'SEASHR') THEN
         QTK = QARG * QFACT((IHOUR+(ISEAS-1)*24),ISRC)
      ELSE IF (QFLAG(ISRC). EQ. 'SHRDOW') THEN
         QTK = QARG * QFACT((IHOUR+(ISEAS-1)*24+
     &        (IDAY_OF_WEEK-1)*96),ISRC)
      END IF

      RETURN
      END

      SUBROUTINE WSADJ
C***********************************************************************
C                 WSADJ Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Adjusts Wind Speed from Anemometer Height to Stack Height
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        INPUTS:  Arrays of Source Parameters
C                 Meteorological Variables for One Hour
C                 Wind Speed Profile Exponents (Default or User-defined)
C
C        OUTPUTS: Stack Top Wind Speed, US
C
C        CALLED FROM:   PCALC
C                       VCALC
C                       ACALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'WSADJ'

C     Adjust Wind Speed -- Assume Wind Speed Constant Below 10 meters
      IF (HS .GE. 10.0) THEN
         US = UREF * (HS/ZREF)**P
      ELSE IF (ZREF .GT. 10.0) THEN
         US = UREF * (10.0/ZREF)**P
      ELSE
         US = UREF
      END IF

C     Do Not Allow Stack Height Wind Speed < 1.0 m/s
      IF (US .LT. 1.0) THEN
         US = 1.0
      END IF

      RETURN
      END

      SUBROUTINE DISTF
C***********************************************************************
C                 DISTF Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Distance to Final Plume Rise
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Buoyancy and Momentum Fluxes
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C
C        OUTPUTS: Distance to Final Plume Rise, XF (m), and Distance
C                 to Final Buoyant Rise (XFB) and Final Momentum Rise (XFM)
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'DISTF'

      IF (UNSTAB .OR. NEUTRL) THEN
         IF (FB .GE. 55.) THEN
            XFB = 119. * FB**0.4
         ELSE IF (FB .GT. 0.) THEN
            XFB = 49. * FB**0.625
         ELSE
            XFB = 4.*DS*(VS+3.*US)*(VS+3.*US)/(VS*US)
         END IF
         XFM = 4.*DS*(VS+3.*US)*(VS+3.*US)/(VS*US)
         XF = MAX(XFB,XFM)
      ELSE IF (STABLE) THEN
         XFB = 2.0715*US/RTOFS
         XFM = 0.5*PI*US/RTOFS
         XF = MAX(XFB,XFM)
      END IF

      RETURN
      END

      SUBROUTINE WAKFLG
C***********************************************************************
C                 WAKFLG Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: To Set Wake Flags for Building Downwash Algorithms
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Building Dimensions
C                 Source Parameters
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Logical Flags for Wake Switches, WAKE and WAKESS;
C                 And Building Types, TALL, SQUAT, and SSQUAT;
C                 And Value of ZLB
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: X2BH

C     Variable Initializations
      MODNAM = 'WAKFLG'

C     Set Initial Wake Switches Based on Building Dimensions
      IF (DSBH.EQ.0.0 .OR. DSBW.EQ.0.0 .OR.
     &    HS .GT. (DSBH + 1.5*MIN(DSBH,DSBW))) THEN
         WAKE   = .FALSE.
         WAKESS = .FALSE.
      ELSE IF (HS .GT. (DSBH + 0.5*MIN(DSBH,DSBW))) THEN
         WAKE   = .TRUE.
         WAKESS = .FALSE.
      ELSE
         WAKE   = .TRUE.
         WAKESS = .TRUE.
      END IF

C     Set Final Wake Switches Based on Plume Height
      IF (WAKE) THEN
         X2BH = DSBH + DSBH
C        Calculate Gradual Momentum Rise at X2BH            ---   CALL DHPMOM
         CALL DHPMOM(X2BH)
         HEMWAK = HS + DHPM
         IF (WAKESS) THEN
            IF (HEMWAK .LE. (DSBH + 2.0*MIN(DSBH,DSBW))) THEN
               WAKE   = .TRUE.
            ELSE
               WAKE   = .FALSE.
               WAKESS = .FALSE.
            END IF
         ELSE
            IF (HEMWAK .LE. (DSBH + 1.5*MIN(DSBH,DSBW))) THEN
               WAKE = .TRUE.
            ELSE
               WAKE = .FALSE.
            END IF
         END IF
      ELSE
         HEMWAK = 0.0
      END IF

C     Set Value of ZLB And Set Logical Flags for Building Type
      IF (WAKE) THEN
         ZLB = MIN(DSBH,DSBW)
         IF (DSBW .LT. DSBH) THEN
C           Tall Building
            TALL  = .TRUE.
            SQUAT = .FALSE.
            SSQUAT= .FALSE.
         ELSE IF (DSBW .LE. 5.*DSBH) THEN
C           Squat Building
            TALL  = .FALSE.
            SQUAT = .TRUE.
            SSQUAT= .FALSE.
         ELSE
C           Super-Squat Building
            TALL  = .FALSE.
            SQUAT = .FALSE.
            SSQUAT= .TRUE.
         END IF
      ELSE
         ZLB = 0.0
      END IF

      RETURN
      END

      SUBROUTINE XYDIST(INDX)
C***********************************************************************
C                 XYDIST Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Sets Receptor Variables and Calculates Downwind (X)
C                 and Crosswind (Y) Distances,
C                 and Radial Distance from Source to Receptor (DISTR)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED BY R.W. Brode, PES, Inc. to use calling argument to
C                 specify array index, so that routine can be used by
C                 both the regular ISCST3 routines and the routines of
C                 the EVENT processor (ISCEV3). - 12/29/97
C
C        INPUTS:  Source Location
C                 Arrays of Receptor Locations
C                 SIN and COS of Wind Direction FROM Which Wind
C                 is Blowing, WDSIN and WDCOS
C
C        OUTPUTS: Values of X, Y, and DISTR (m)
C
C        CALLED FROM:   PCALC
C                       VCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: INDX

C     Variable Initializations
      MODNAM = 'XYDIST'

C     Set Receptor Coordinates, Terrain Elevation and Flagpole Heights
      XR = AXR(INDX)
      YR = AYR(INDX)
      ZELEV = AZELEV(INDX)
      ZFLAG = AZFLAG(INDX)

C     Calculate Downwind (X) and Crosswind (Y) Distances
      X = -((XR-XS)*WDSIN + (YR-YS)*WDCOS)
      Y =   (XR-XS)*WDCOS - (YR-YS)*WDSIN

C     Calculate Source-Receptor (Radial) Distance, DISTR
      DISTR = SQRT (X*X + Y*Y)

      RETURN
      END

      SUBROUTINE DECAY (XARG)
C***********************************************************************
C                 DECAY Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Decay Term for Use in Gaussian Plume Equation
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance, XARG (m)
C                 Stack Top Wind Speed, US (m/s)
C                 Decay Coefficient, DECOEFF (1/s)
C
C        OUTPUTS: Decay Term, D
C
C        CALLED FROM:   CHI
C                       DEP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: XARG

C     Variable Initializations
      MODNAM = 'DECAY'

      D = 1.0

      IF (DECOEF .GT. 0.0) THEN
         D = EXP (MAX (EXPLIM, -DECOEF*XARG/US))
      END IF

      RETURN
      END

      SUBROUTINE VERT(HEARG,SZARG,A0,ZARG,VOUT)
C***********************************************************************
C                 VERT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Vertical Term for Use in Gaussian Plume Equation
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED BY R.W. Brode, PES, Inc. to include check for HEARG > ZI,
C                 and to incorporate option to correct problem.  02/04/2002
C
C        MODIFIED BY R.W. Brode, PES, Inc. to set vertical term to 0.0
C                 for cases when receptor is above mixing height - 12/29/97
C
C        MODIFIED BY R.W. Brode, PES, Inc. to use calling arguments - 9/30/94
C
C        MODIFIED BY D. Strimaitis, SRC (for Wet REMOVAL of Gases)
C
C        DATE:    November 8, 1993
C
C
C        INPUTS:  Plume Height (m), HEARG
C                 Vertical Dispersion Parameter (m), SZARG
C                 Stability Class, KST
C                 Mixing Height (m), ZI
C                 Receptor Height Above Ground (m), ZARG
C
C        OUTPUTS: Vertical Term, VOUT
C
C        CALLED FROM:   PSIMPL, PCOMPL, ASIMPL
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I
      REAL    :: VOUT, ZARG, A0, SZARG, HEARG, A1, A2, A3, A4, A5, A6,
     &           TWOIZI, SUM, T, HESAV

C     Variable Initializations
      MODNAM = 'VERT'

      VOUT = 0.0

      HESAV = HEARG
C     Check for HEARG > ZI
      IF (.NOT.STABLE .AND. HEARG .GT. ZI) THEN
C        WRITE Informational Message
         WRITE(DUMMY,'(2I4)') ISRC, IREC
         CALL ERRHDL(PATH,MODNAM,'I','283',DUMMY)
         IF (HEGTZI) THEN
C           Non-default option to adjust vertical term used, set HE=ZI.
            HEARG = ZI
         END IF
      END IF

      IF (ZARG .EQ. 0.0) THEN
C        Vertical Term for Case With No Flagpole Receptor
         IF (STABLE .OR. ZI.GE.10000.) THEN
            A1 = A0 * HEARG * HEARG
            IF (A1 .GT. EXPLIM)  VOUT = 2.*EXP(A1)
         ELSE IF ((SZARG/ZI) .GE. 1.6) THEN
            VOUT  = SRT2PI*(SZARG/ZI)
         ELSE
            A1 = A0 * HEARG * HEARG
            IF (A1 .GT. EXPLIM)  VOUT = EXP(A1)
            SUM = 0.0
            DO I = 1, 100
               T  = 0.0
               TWOIZI = 2.*I*ZI
               A2 = A0 * (TWOIZI-HEARG) * (TWOIZI-HEARG)
               A3 = A0 * (TWOIZI+HEARG) * (TWOIZI+HEARG)
               IF (A2 .GT. EXPLIM)  T = EXP(A2)
               IF (A3 .GT. EXPLIM)  T = T + EXP(A3)
               SUM = SUM + T
               IF (ABS(T) .LE. 5.0E-9) THEN
C                 Exit Loop
                  EXIT
               END IF
            END DO
C           Calculate Total Vert. Term - (2.*) was Removed for Optimization
            VOUT  = 2.*(VOUT + SUM)
         END IF
      ELSE IF (ZARG .LE. ZI) THEN
C        Vertical Term for Case of ZARG .NE. 0.0
         IF (STABLE .OR. ZI .GE. 10000.) THEN
            A1 = A0 * (ZARG-HEARG) * (ZARG-HEARG)
            A2 = A0 * (ZARG+HEARG) * (ZARG+HEARG)
            IF (A1 .GT. EXPLIM)  VOUT = EXP(A1)
            IF (A2 .GT. EXPLIM)  VOUT = VOUT + EXP(A2)
         ELSE IF (SZARG/ZI .GE. 1.6) THEN
            VOUT  = SRT2PI*(SZARG/ZI)
         ELSE
            A1 = A0 * (ZARG-HEARG) * (ZARG-HEARG)
            A2 = A0 * (ZARG+HEARG) * (ZARG+HEARG)
            IF (A1 .GT. EXPLIM)  VOUT = EXP(A1)
            IF (A2 .GT. EXPLIM)  VOUT = VOUT + EXP(A2)
            SUM = 0.0
            DO I = 1, 100
               T  = 0.0
               TWOIZI = 2.*I*ZI
               A3 = A0 * (ZARG-(TWOIZI-HEARG)) * (ZARG-(TWOIZI-HEARG))
               A4 = A0 * (ZARG+(TWOIZI-HEARG)) * (ZARG+(TWOIZI-HEARG))
               A5 = A0 * (ZARG-(TWOIZI+HEARG)) * (ZARG-(TWOIZI+HEARG))
               A6 = A0 * (ZARG+(TWOIZI+HEARG)) * (ZARG+(TWOIZI+HEARG))
               IF (A3 .GT. EXPLIM)  T = T + EXP(A3)
               IF (A4 .GT. EXPLIM)  T = T + EXP(A4)
               IF (A5 .GT. EXPLIM)  T = T + EXP(A5)
               IF (A6 .GT. EXPLIM)  T = T + EXP(A6)
               SUM = SUM + T
               IF (ABS(T) .LE. 1.0E-8) THEN
C                 Exit Loop
                  EXIT
               END IF
            END DO
            VOUT  = VOUT + SUM
         END IF
      ELSE
C       Receptor is above mixing height, set VOUT=0.
        VOUT = 0.0
      END IF

      HEARG = HESAV

      RETURN
      END

      SUBROUTINE SUMVAL
C***********************************************************************
C                 SUMVAL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Sums HRVAL to AVEVAL and ANNVAL Arrays
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  HRVAL - Hourly Value for (IREC,ISRC) Combination
C                 Averaging Period Options
C                 Source Groupings
C
C        OUTPUTS: Updated Sums of AVEVAL and ANNVAL Arrays
C
C        CALLED FROM:   PCALC
C                       VCALC
C                       ACALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'SUMVAL'

C     Begin LOOP Over Output Types
      DO ITYP = 1, NUMTYP
         IF (HRVAL(ITYP) .NE. 0.0) THEN
C           Begin Source Group LOOP
            DO IGRP = 1, NUMGRP
C              Check for Source Belonging to Group
               IF (IGROUP(ISRC,IGRP) .EQ. 1) THEN
C                 Begin Averaging Period LOOP
                  DO IAVE = 1, NUMAVE
                     AVEVAL(IREC,IGRP,IAVE,ITYP) = HRVAL(ITYP) +
     &                                       AVEVAL(IREC,IGRP,IAVE,ITYP)
                  END DO
C                 End Averaging Period LOOP
                  IF (PERIOD .OR. ANNUAL) THEN
                     IF (.NOT.SCIM .OR. (SCIM.AND..NOT.WETSCIM)) THEN
                        ANNVAL(IREC,IGRP,ITYP)
     &                              = HRVAL(ITYP) +
     &                                ANNVAL(IREC,IGRP,ITYP)
                     END IF
                     IF (SCIM .AND. WETSCIM .AND. WETHR) THEN
                        ANNVALW(IREC,IGRP,ITYP)
     &                                 = HRVAL(ITYP) +
     &                                   ANNVALW(IREC,IGRP,ITYP)
                     END IF
                     IF (SCIM .AND. WETSCIM .AND. SCIMHR) THEN
                        ANNVALD(IREC,IGRP,ITYP)
     &                              = HRVALD(ITYP) +
     &                                ANNVALD(IREC,IGRP,ITYP)
                     END IF
                  END IF
                  IF (ISEAHR(IGRP) .EQ. 1) THEN
                     SHVALS(IREC,IGRP,ISEAS,IHOUR,ITYP) = HRVAL(ITYP) +
     &               SHVALS(IREC,IGRP,ISEAS,IHOUR,ITYP)
                  END IF
               END IF
            END DO
C           End Source Group LOOP
         END IF
      END DO
C     End LOOP Over Output Types

      RETURN
      END

      SUBROUTINE AVER
C***********************************************************************
C                 AVER Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Short Term (<=24 hr) Average Concentrations
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Averaging Time Option Switches
C                 Updated Array of Cumulative Values, AVEVAL
C
C        OUTPUTS: Updated Array of Averages, AVEVAL
C
C        CALLED FROM: HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: SNUM

C     Variable Initializations
      MODNAM = 'AVER'

      IF (KAVE(IAVE) .NE. 1) THEN
C        Calculate Denominator Considering Calms and Missing,
C        Skipping Averaging if Averaging Period is 1-Hour
         SNUM = AMAX0((NUMHRS(IAVE)-NUMCLM(IAVE)-NUMMSG(IAVE)),
     &                 NINT(NUMHRS(IAVE)*0.75+0.4))
C        Begin Source Group LOOP
         DO IGRP = 1, NUMGRP
C           Begin Receptor LOOP
            DO IREC = 1, NUMREC
               AVEVAL(IREC,IGRP,IAVE,1) = (1./SNUM)*
     &                                  AVEVAL(IREC,IGRP,IAVE,1)
            END DO
C           End Receptor LOOP
         END DO
C        End Source Group LOOP
      END IF

      RETURN
      END

      SUBROUTINE HIVALS
C***********************************************************************
C                 HIVALS Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Updates High Value Tables
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To change subroutine name MAXVAL to MAXVALUE to
C                    avoid conflicts with intrinsic function MAXVAL under
C                    Fortran 90.  R. Brode, PES, 12/29/97
C
C        INPUTS:  High Value Option Switches
C                 Array of CONC or DEPOS Averages
C
C        OUTPUTS: Updated High Value Arrays
C
C        CALLED FROM:   MAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'HIVALS'

C     Check for High/Max Value Options - Skip Update If KAVE=1,
C     And No CALCS Were Made for the Current Hour
      IF (CALCS .OR. KAVE(IAVE).NE.1) THEN
         IF (INHI(IAVE) .EQ. 1) THEN
            DO ITYP = 1, NUMTYP
C              Update High Values for Each Receptor            ---   CALL NHIGH
               CALL NHIGH
            END DO
         END IF
         IF (MAXAVE(IAVE) .EQ. 1) THEN
            DO ITYP = 1, NUMTYP
C              Update Maximum Value Table for KAVE             ---   CALL MAXVALUE
               CALL MAXVALUE
            END DO
         END IF
      END IF
C     Reset Counters for This Averaging Period
      NUMHRS(IAVE) = 0
      NUMCLM(IAVE) = 0
      NUMMSG(IAVE) = 0

      RETURN
      END

      SUBROUTINE NHIGH
C***********************************************************************
C                 NHIGH Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Update Highest Value by Receptor Arrays
C                 NVAL = 6 Assigned in PARAMETER Statement
C                 Note: For duplicate values, the earlier occurrence keeps its
C                       rank within the array
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  High Value Options
C                 Array of CONC or DEPOS Averages
C                 Averaging Period
C
C        OUTPUTS: Updated Highest Value Array
C                 Updated Highest Date Array
C
C        CALLED FROM:   HIVALS
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J

C     Variable Initializations
      MODNAM = 'NHIGH'

C     Begin Source Group LOOP
      DO IGRP = 1, NUMGRP
C        Begin Receptor LOOP
         DO IREC = 1, NUMREC
            IF (NHIVAL .GT. 1) THEN
               IF (AVEVAL(IREC,IGRP,IAVE,ITYP) .GT.
     &                    HIVALU(IREC,NHIVAL,IGRP,IAVE,ITYP)) THEN
                  DO J = NHIVAL-1, 1, -1
                     IF (AVEVAL(IREC,IGRP,IAVE,ITYP) .LE.
     &                          HIVALU(IREC,J,IGRP,IAVE,ITYP)) THEN
                        HIVALU(IREC,J+1,IGRP,IAVE,ITYP) =
     &                      AVEVAL(IREC,IGRP,IAVE,ITYP)
                        IF (NUMCLM(IAVE).EQ.0 .AND.
     &                      NUMMSG(IAVE).EQ.0) THEN
                           HCLMSG(IREC,J+1,IGRP,IAVE,ITYP) = ' '
                        ELSE
C                          Set Indicator Of Calm and Msg    ---   CALL HSETFG
                           CALL HSETFG(0,J)
                        END IF
                        NHIDAT(IREC,J+1,IGRP,IAVE,ITYP) = KURDAT
C                       Exit Block
                        GO TO 200
                     ELSE
                        HIVALU(IREC,J+1,IGRP,IAVE,ITYP) =
     &                    HIVALU(IREC,J,IGRP,IAVE,ITYP)
                        HCLMSG(IREC,J+1,IGRP,IAVE,ITYP) =
     &                    HCLMSG(IREC,J,IGRP,IAVE,ITYP)
                        NHIDAT(IREC,J+1,IGRP,IAVE,ITYP) =
     &                    NHIDAT(IREC,J,IGRP,IAVE,ITYP)
                        IF (J .EQ. 1) THEN
                           HIVALU(IREC,1,IGRP,IAVE,ITYP) =
     &                       AVEVAL(IREC,IGRP,IAVE,ITYP)
                           IF (NUMCLM(IAVE).EQ.0 .AND.
     &                         NUMMSG(IAVE).EQ.0) THEN
                              HCLMSG(IREC,1,IGRP,IAVE,ITYP) = ' '
                           ELSE
C                             Set Indicator Of Calm and Msg ---   CALL HSETFG
                              CALL HSETFG(1,1)
                           END IF
                           NHIDAT(IREC,1,IGRP,IAVE,ITYP) = KURDAT
                        END IF
                     END IF
                  END DO
               END IF
            ELSE IF (NHIVAL .EQ. 1) THEN
               IF (AVEVAL(IREC,IGRP,IAVE,ITYP) .GT.
     &                    HIVALU(IREC,1,IGRP,IAVE,ITYP)) THEN
             HIVALU(IREC,1,IGRP,IAVE,ITYP) = AVEVAL(IREC,IGRP,IAVE,ITYP)
                  IF (NUMCLM(IAVE).EQ.0 .AND.
     &                NUMMSG(IAVE).EQ.0) THEN
                     HCLMSG(IREC,1,IGRP,IAVE,ITYP) = ' '
                  ELSE
C                    Set Indicator Of Calm and Missing      ---   CALL HSETFG
                     CALL HSETFG(1,1)
                  END IF
                  NHIDAT(IREC,1,IGRP,IAVE,ITYP) = KURDAT
               END IF
            END IF
 200        CONTINUE
C        End Receptor LOOP
         END DO
C     End Source Group LOOP
      END DO

      RETURN
      END

      SUBROUTINE HSETFG(INDT,J)
C***********************************************************************
C                 HSETFG Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Set Calm and Missing Flag Of the Result
C
C        PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To correct error in order of indices for array
C                    HCLMSG on first assignment to 'b' - 9/29/92
C
C        INPUTS:  High Value Options
C                 Array of CONC or DEPOS Averages
C                 Averaging Period
C
C        OUTPUTS: Updated Highest Value Flag Array
C
C        CALLED FROM:   NHIGH
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J, INDT

C     Variable Initializations
      MODNAM = 'HSETFG'

      IF (INDT .EQ. 0) THEN
C        Set Indicator Of Calm and Missing
         IF (NUMCLM(IAVE).NE.0 .AND.
     &       NUMMSG(IAVE).EQ.0) THEN
             HCLMSG(IREC,J+1,IGRP,IAVE,ITYP) = 'c'
         ELSE IF (NUMCLM(IAVE).EQ.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             HCLMSG(IREC,J+1,IGRP,IAVE,ITYP) = 'm'
         ELSE IF (NUMCLM(IAVE).NE.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             HCLMSG(IREC,J+1,IGRP,IAVE,ITYP) = 'b'
         END IF
      ELSE IF (INDT .EQ. 1) THEN
C        Set Indicator Of Calm and Missing
         IF (NUMCLM(IAVE).NE.0 .AND.
     &       NUMMSG(IAVE).EQ.0) THEN
             HCLMSG(IREC,1,IGRP,IAVE,ITYP) = 'c'
         ELSE IF (NUMCLM(IAVE).EQ.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             HCLMSG(IREC,1,IGRP,IAVE,ITYP) = 'm'
         ELSE IF (NUMCLM(IAVE).NE.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             HCLMSG(IREC,1,IGRP,IAVE,ITYP) = 'b'
         END IF
      END IF

      RETURN
      END

      SUBROUTINE MAXVALUE
C***********************************************************************
C                 MAXVALUE Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Update Overall Maximum Value Arrays
C                 NMAX = 50 Assigned in PARAMETER Statement
C                 Note: For duplicate values, the earlier occurrence keeps
C                       its rank within the array
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Maximum Value Table Options
C                 Array of CONC or DEPOS Averages
C                 Averaging Period
C
C        OUTPUTS: Updated Maximum Value Array
C                 Updated Maximum Date Array
C                 Updated Maximum Receptor Array
C
C        CALLED FROM:   HIVALS
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J

C     Variable Initializations
      MODNAM = 'MAXVALUE'

C     Begin Source Group LOOP
      DO IGRP = 1, NUMGRP
C        Begin Receptor LOOP
         DO IREC = 1, NUMREC
            IF (NMXVAL .GT. 1) THEN
               IF (AVEVAL(IREC,IGRP,IAVE,ITYP) .GT.
     &                       RMXVAL(NMXVAL,IGRP,IAVE,ITYP)) THEN
                  DO J = NMXVAL-1, 1, -1
                     IF(AVEVAL(IREC,IGRP,IAVE,ITYP) .LE.
     &                     RMXVAL(J,IGRP,IAVE,ITYP)) THEN
                RMXVAL(J+1,IGRP,IAVE,ITYP) = AVEVAL(IREC,IGRP,IAVE,ITYP)
                        IF (NUMCLM(IAVE).EQ.0 .AND.
     &                      NUMMSG(IAVE).EQ.0) THEN
                           MCLMSG(J+1,IGRP,IAVE,ITYP) = ' '
                        ELSE
C                          Set Indicator Of Calm and Msg    ---   CALL MSETFG
                           CALL MSETFG(0,J)
                        END IF
                        MXDATE(J+1,IGRP,IAVE,ITYP) = KURDAT
                        MXLOCA(J+1,IGRP,IAVE,ITYP) = IREC
C                       Exit Block
                        GO TO 200
                     ELSE
                   RMXVAL(J+1,IGRP,IAVE,ITYP) = RMXVAL(J,IGRP,IAVE,ITYP)
                   MXDATE(J+1,IGRP,IAVE,ITYP) = MXDATE(J,IGRP,IAVE,ITYP)
                   MCLMSG(J+1,IGRP,IAVE,ITYP) = MCLMSG(J,IGRP,IAVE,ITYP)
                   MXLOCA(J+1,IGRP,IAVE,ITYP) = MXLOCA(J,IGRP,IAVE,ITYP)
                        IF (J .EQ. 1) THEN
                  RMXVAL(1,IGRP,IAVE,ITYP) = AVEVAL(IREC,IGRP,IAVE,ITYP)
                           IF (NUMCLM(IAVE).EQ.0 .AND.
     &                         NUMMSG(IAVE).EQ.0) THEN
                              MCLMSG(1,IGRP,IAVE,ITYP) = ' '
                           ELSE
C                             Set Indicator Of Calm and Msg ---   CALL MSETFG
                              CALL MSETFG(1,1)
                           END IF
                           MXDATE(1,IGRP,IAVE,ITYP) = KURDAT
                           MXLOCA(1,IGRP,IAVE,ITYP) = IREC
                        END IF
                     END IF
                   END DO
               END IF
            ELSE IF (NMXVAL .EQ. 1) THEN
               IF (AVEVAL(IREC,IGRP,IAVE,ITYP) .GT.
     &                RMXVAL(1,IGRP,IAVE,ITYP)) THEN
                  RMXVAL(1,IGRP,IAVE,ITYP) = AVEVAL(IREC,IGRP,IAVE,ITYP)
                  IF (NUMCLM(IAVE).EQ.0 .AND.
     &                NUMMSG(IAVE).EQ.0) THEN
                     MCLMSG(1,IGRP,IAVE,ITYP) = ' '
                  ELSE
C                    Set Indicator Of Calm and Missing      ---   CALL MSETFG
                     CALL MSETFG(1,1)
                  END IF
                  MXDATE(1,IGRP,IAVE,ITYP) = KURDAT
                  MXLOCA(1,IGRP,IAVE,ITYP) = IREC
               END IF
            END IF
 200        CONTINUE
         END DO
C        End Receptor LOOP
      END DO
C     End Source Group LOOP

      RETURN
      END

      SUBROUTINE MSETFG(INDT,J)
C***********************************************************************
C                 MSETFG Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Set Calm and Missing Flag Of the Max Result
C
C        PROGRAMMER: Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Maximum Value Table Options
C                 Array of CONC or DEPOS Averages
C                 Averaging Period
C
C        OUTPUTS: Updated Maximum Value Flag Array
C
C        CALLED FROM:   MAXVALUE
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J, INDT

C     Variable Initializations
      MODNAM = 'MSETFG'

      IF (INDT .EQ. 0) THEN
C        Set Indicator Of Calm and Missing
         IF (NUMCLM(IAVE).NE.0 .AND.
     &       NUMMSG(IAVE).EQ.0) THEN
             MCLMSG(J+1,IGRP,IAVE,ITYP) = 'c'
         ELSE IF (NUMCLM(IAVE).EQ.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             MCLMSG(J+1,IGRP,IAVE,ITYP) = 'm'
         ELSE IF (NUMCLM(IAVE).NE.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             MCLMSG(J+1,IGRP,IAVE,ITYP) = 'b'
         END IF
      ELSE IF (INDT .EQ. 1) THEN
C        Set Indicator Of Calm and Missing
         IF (NUMCLM(IAVE).NE.0 .AND.
     &       NUMMSG(IAVE).EQ.0) THEN
             MCLMSG(1,IGRP,IAVE,ITYP) = 'c'
         ELSE IF (NUMCLM(IAVE).EQ.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             MCLMSG(1,IGRP,IAVE,ITYP) = 'm'
         ELSE IF (NUMCLM(IAVE).NE.0 .AND.
     &            NUMMSG(IAVE).NE.0) THEN
             MCLMSG(1,IGRP,IAVE,ITYP) = 'b'
         END IF
      END IF

      RETURN
      END

      SUBROUTINE MAXFIL
C***********************************************************************
C                 MAXFIL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Update Maximum Value File (>Threshold)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   Moved check for RSTSAV (SAVEFILE option) outside
C                    the receptor loop, and replaced 'read to end' loop
C                    with POSITION='APPEND' in OPEN statement for
C                    Fortran 90 version.
C                    R.W. Brode, PES, Inc.,  6/23/98
C
C        INPUTS:  Maximum File Options
C                 Array of CONC or DEPOS Averages
C                 Averaging Period
C
C        OUTPUTS: Updated Maximum Value File
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'MAXFIL'

C     Check for High/Max Value Options - Skip Update If KAVE=1,
C     And No CALCS Were Made for the Current Hour
      IF (CALCS .OR. KAVE(IAVE).NE.1) THEN
C        Begin Source Group LOOP
         DO IGRP = 1, NUMGRP
C           Check for MAXIFILE Option for This IGRP,IAVE Combination
            IF (MAXFLE(IGRP,IAVE) .EQ. 1) THEN
C              Begin Receptor LOOP
               DO IREC = 1, NUMREC
C                 For the Values Over Threshold
                  IF (AVEVAL(IREC,IGRP,IAVE,1) .GE.
     &                     THRESH(IGRP,IAVE)) THEN
                     WRITE(IMXUNT(IGRP,IAVE),THRFRM,ERR=99) KAVE(IAVE),
     &                  GRPID(IGRP), KURDAT, AXR(IREC), AYR(IREC),
     &                  AZELEV(IREC), AZFLAG(IREC),
     &                  AVEVAL(IREC,IGRP,IAVE,1)
                  END IF
               END DO
C              End Receptor LOOP
               IF (RSTSAV) THEN
C                 Saving Intermediate Results to File for Later Re-start
C                 Close MAXIFILE and Reposition to End
                  CLOSE (IMXUNT(IGRP,IAVE))
                  OPEN(IMXUNT(IGRP,IAVE),FILE=THRFIL(IGRP,IAVE),
     &                 POSITION='APPEND')
               END IF
            END IF
         END DO
C        End Source Group LOOP
      END IF

      GO TO 999

C     WRITE Error Message for Problem Writing to Maximum Value File
 99   WRITE(DUMMY,'("MAXFL",I3.3)') IMXUNT(IGRP,IAVE)
      CALL ERRHDL(PATH,MODNAM,'E','520',DUMMY)
      RUNERR = .TRUE.

 999  RETURN
      END

      SUBROUTINE POSTFL
C***********************************************************************
C                 POSTFL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Write Concurrent Values to File for Postprocessing
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   Replaced 'read to end' loop with POSITION='APPEND'
C                    in OPEN statements for Fortran 90 version with
C                    RSTSAV (SAVEFILE option).
C                    R.W. Brode, PES, Inc.,  6/23/98
C
C        INPUTS:  Postprocessing File Options
C                 Array of CONC or DEPOS Averages
C
C        OUTPUTS: Postprocessor Files
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'POSTFL'

C     Begin Source Group LOOP
      DO IGRP = 1, NUMGRP
C        Check for POSTFILE Option for This IGRP,IAVE Combination
         IF (IPSTFL(IGRP,IAVE) .EQ. 1) THEN
            IF (IPSFRM(IGRP,IAVE) .EQ. 0) THEN
C              WRITE Results to Unformatted POSTFILE
               WRITE(IPSUNT(IGRP,IAVE),ERR=99) KURDAT, KAVE(IAVE),
     &            GRPID(IGRP), ((AVEVAL(IREC,IGRP,IAVE,ITYP),
     &                           IREC=1,NUMREC),ITYP=1,NUMTYP)
               IF (RSTSAV) THEN
C                 Saving Intermediate Results to File for Later Re-start
C                 Close POSTFILE and Reposition to End
                  CLOSE (IPSUNT(IGRP,IAVE))
                  OPEN(IPSUNT(IGRP,IAVE),FILE=PSTFIL(IGRP,IAVE),
     &                 FORM='UNFORMATTED',POSITION='APPEND')
               END IF
            ELSE
C              WRITE Results to Formatted Plot File
C              Begin Receptor LOOP
               DO IREC = 1, NUMREC
                  WRITE(IPSUNT(IGRP,IAVE),PSTFRM,ERR=99)
     &               AXR(IREC), AYR(IREC), (AVEVAL(IREC,IGRP,IAVE,ITYP),
     &                                      ITYP=1,NUMTYP),
     &               AZELEV(IREC), CHRAVE(IAVE), GRPID(IGRP), KURDAT,
     &               NETID(IREC)
               END DO
C              End Receptor LOOP
               IF (RSTSAV) THEN
C                 Saving Intermediate Results to File for Later Re-start
C                 Close POSTFILE and Reposition to End
                  CLOSE (IPSUNT(IGRP,IAVE))
                  OPEN(IPSUNT(IGRP,IAVE),FILE=PSTFIL(IGRP,IAVE),
     &                 FORM='FORMATTED',POSITION='APPEND')
               END IF
            END IF
         END IF
      END DO
C     End Source Group LOOP

      GO TO 999

C     WRITE Error Message for Problem Writing to Postprocessor File
 99   WRITE(DUMMY,'("PSTFL",I3.3)') IPSUNT(IGRP,IAVE)
      CALL ERRHDL(PATH,MODNAM,'E','520',DUMMY)
      RUNERR = .TRUE.

 999  RETURN
      END

      SUBROUTINE TOXXFL
C***********************************************************************
C                 TOXXFL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Update TOXXFILE Buffers, and Write Out if Full
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    September 29, 1992
C
C        INPUTS:  TOXXFILE Options
C                 Array of CONC or DEPOS Averages
C                 Averaging Period
C
C        OUTPUTS: Updated TOXXFILE Buffers and File
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, IG, ICODE
      REAL    :: CUTOFF

C     Variable Initializations
      MODNAM = 'TOXXFL'

C     Check for TOXXFILE Option - Skip Update If KAVE=1,
C     And No CALCS Were Made for the Current Hour
      IF (ITOXFL(IAVE).EQ.1 .AND. (CALCS .OR. KAVE(IAVE).NE.1)) THEN
C        Convert TOXXFILE Threshold to User Units
         CUTOFF = TOXTHR(IAVE) * EMIFAC(1)

C        Begin Receptor LOOP
         DO IREC = 1, NUMREC

C           Begin Source Group LOOP
            DO IGRP = 1, NUMGRP

C              For the Values Over Threshold (in user units), Fill Buffers
               IF (AVEVAL(IREC,IGRP,IAVE,1) .GE. CUTOFF) THEN
                  DO IG = 1, NUMGRP
C                    Loop Through Groups and Write Values to Buffer
                     IPAIR = IPAIR + 1
                     ICODE = 100000*ILINE + 1000*IG + IREC
                     IDCONC(IAVE,IPAIR) = ICODE
C                    Convert CONC Values Back to Units of g/s
                     TXCONC(IAVE,IPAIR)=AVEVAL(IREC,IG,IAVE,1)/EMIFAC(1)
                     IF (IPAIR .EQ. NPAIR) THEN
C                       Write Out Full Buffers and Reset Counter
                        WRITE(ITXUNT(IAVE),ERR=99) (IDCONC(IAVE,I),
     &                                              I=1,NPAIR)
                        WRITE(ITXUNT(IAVE),ERR=99) (TXCONC(IAVE,I),
     &                                              I=1,NPAIR)
                        IPAIR = 0
                     END IF
                  END DO
C                 Exit Source Group LOOP
                  EXIT
               END IF

            END DO
C           End Source Group LOOP

         END DO
C        End Receptor LOOP
      END IF

      GO TO 999

C     WRITE Error Message for Problem Writing to TOXXFILE
 99   WRITE(DUMMY,'("TOXFL",I3.3)') ITXUNT(IAVE)
      CALL ERRHDL(PATH,MODNAM,'E','520',DUMMY)
      RUNERR = .TRUE.

 999  RETURN
      END

      SUBROUTINE PRTDAY
C***********************************************************************
C                 PRTDAY Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Write Concurrent Values to Printed Output File
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To adjust format statement 9082 for BOUNDARY receptors
C                    to better accommodate UTM coordinates - 9/29/92
C
C        INPUTS:  Postprocessing File Options
C                 Array of CONC or DEPOS Averages
C
C        OUTPUTS: Postprocessor Files
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, J, K, II, NX, NY, INDZ, INDC, ISRF
      REAL    :: YCOVAL, XRMS, YRMS, DIST, DIR
      CHARACTER BUF132*132

C     Variable Initializations
      MODNAM = 'PRTDAY'

C     Begin Source Group LOOP
      DO IGRP = 1, NUMGRP

C        Fill Work Array With SRCIDs For This Group
         INDGRP = 0
         DO ISRC = 1, NUMSRC
            IF (IGROUP(ISRC,IGRP) .EQ. 1) THEN
               INDGRP = INDGRP + 1
               WORKID(INDGRP) = SRCID(ISRC)
            END IF
         END DO
C        Check for More Than 31 Sources Per Group
         IF (INDGRP .GT. 31) THEN
            WORKID(31) = ' . . . '
            INDGRP = 31
         END IF

C        Print Results for Receptor Networks
C        Set Number of Columns Per Page, NCPP
         NCPP = 9
C        Set Number of Rows Per Page, NRPP
         NRPP = 40
C        Begin LOOP Through Networks
         DO I = 1, INNET
C           Calculate Number of Pages Per X-Group, NPPX, & Per Y-Group, NPPY
            NPPX = 1 + INT((NUMXPT(I)-1)/NCPP)
            NPPY = 1 + INT((NUMYPT(I)-1)/NRPP)
            DO NX = 1, NPPX
               DO NY = 1, NPPY
                  CALL HEADER
                  WRITE(IOUNIT,9032) CHRAVE(IAVE), (CHIDEP(II,ITYP),
     &                                              II=1,6),
     &               IHOUR,JDAY,IYR,GRPID(IGRP),(WORKID(K),K = 1,INDGRP)
                  WRITE(IOUNIT,9037) NTID(I), NTTYP(I)
C                 Print The Value By Groups
                  WRITE(IOUNIT,9011) CHIDEP(3,ITYP), POLLUT,OUTLBL(ITYP)
                  IF (NX .EQ. NPPX) THEN
                     IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                        WRITE(IOUNIT,9016)
                        WRITE(IOUNIT,9017) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NUMXPT(I))
                     ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                        WRITE(IOUNIT,9018)
                        WRITE(IOUNIT,9019) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NUMXPT(I))
                     END IF
                  ELSE
                     IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                        WRITE(IOUNIT,9016)
                        WRITE(IOUNIT,9017) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NCPP*NX)
                     ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                        WRITE(IOUNIT,9018)
                        WRITE(IOUNIT,9019) (XCOORD(J,I),J=1+NCPP*(NX-1),
     &                                                    NCPP*NX)
                     END IF
                  END IF
                  WRITE(IOUNIT,9010)
                  IF (NY .EQ. NPPY) THEN
                     DO K = 1+NRPP*(NY-1), NUMYPT(I)
                        IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                           INDZ = NETEND(I) - K*NUMXPT(I) + 1
                           YCOVAL = YCOORD(NUMYPT(I)-K+1,I)
                        ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                           INDZ = NETSTA(I) + (K-1)*NUMXPT(I)
                           YCOVAL = YCOORD(K,I)
                        END IF
                        IF (NX .EQ. NPPX) THEN
                           WRITE(IOUNIT,9013) YCOVAL,
     &                 (AVEVAL(INDZ+J-1,IGRP,IAVE,ITYP),J=1+NCPP*(NX-1),
     &                                                  NUMXPT(I))
                        ELSE
                           WRITE(IOUNIT,9013) YCOVAL,
     &                 (AVEVAL(INDZ+J-1,IGRP,IAVE,ITYP),J=1+NCPP*(NX-1),
     &                                                  NCPP*NX)
                        END IF
                     END DO
                  ELSE
                     DO K = 1+NRPP*(NY-1), NRPP*NY
                        IF (NTTYP(I) .EQ. 'GRIDCART') THEN
                           INDZ = NETEND(I) - K*NUMXPT(I) + 1
                           YCOVAL = YCOORD(NUMYPT(I)-K+1,I)
                        ELSE IF (NTTYP(I) .EQ. 'GRIDPOLR') THEN
                           INDZ = NETSTA(I) + (K-1)*NUMXPT(I)
                           YCOVAL = YCOORD(K,I)
                        END IF
                        IF (NX .EQ. NPPX) THEN
                           WRITE(IOUNIT,9013) YCOVAL,
     &                 (AVEVAL(INDZ+J-1,IGRP,IAVE,ITYP),J=1+NCPP*(NX-1),
     &                                                  NUMXPT(I))
                        ELSE
                           WRITE(IOUNIT,9013) YCOVAL,
     &                 (AVEVAL(INDZ+J-1,IGRP,IAVE,ITYP),J=1+NCPP*(NX-1),
     &                                                  NCPP*NX)
                        END IF
                     END DO
                  END IF
               END DO
            END DO
         END DO
C        End LOOP Through Networks

         IF (IRSTAT(4) .NE. 0) THEN
C           Print Out The Coord. & Concentrations For Discrete Cart Receptors
            INDC = 0
            DO IREC = 1, NUMREC
               IF (RECTYP(IREC) .EQ. 'DC') THEN
                  INDC = INDC + 1
                  IF (MOD(INDC-1,80) .EQ. 0) THEN
                     CALL HEADER
                     WRITE(IOUNIT,9032) CHRAVE(IAVE),(CHIDEP(II,ITYP),
     &         II=1,6),IHOUR,JDAY,IYR,GRPID(IGRP),(WORKID(K),K=1,INDGRP)
                     WRITE(IOUNIT,9043)
                     WRITE(IOUNIT,9011) CHIDEP(3,ITYP), POLLUT,
     &                                  OUTLBL(ITYP)
                     WRITE(IOUNIT,9048) CHIDEP(3,ITYP), CHIDEP(3,ITYP)
                  END IF
                  IF (MOD(INDC,2) .NE. 0) THEN
                     WRITE(BUF132(1:60),9045) AXR(IREC),AYR(IREC),
     &                     AVEVAL(IREC,IGRP,IAVE,ITYP)
                  ELSE
                     WRITE(BUF132(61:120),9045) AXR(IREC),
     &                     AYR(IREC), AVEVAL(IREC,IGRP,IAVE,ITYP)
                     WRITE(IOUNIT,9090) BUF132
                     WRITE(BUF132,9095)
                  END IF
               END IF
            END DO
            IF (MOD(INDC,2) .NE. 0) THEN
               WRITE(IOUNIT,9090) BUF132
               WRITE(BUF132,9095)
            END IF
         END IF

         IF (IRSTAT(5) .NE. 0) THEN
C           Print Out The Coord. & Concentrations For Discrete Polar Receptors
            INDC = 0
            DO IREC = 1, NUMREC
               IF (RECTYP(IREC) .EQ. 'DP') THEN
                  INDC = INDC + 1
                  XRMS = AXR(IREC) - AXS(IREF(IREC))
                  YRMS = AYR(IREC) - AYS(IREF(IREC))
                  DIST = SQRT(XRMS*XRMS + YRMS*YRMS)
                  DIR  = ATAN2(XRMS, YRMS) * RTODEG
                  IF (DIR .LE. 0.0) DIR = DIR + 360.
                  IF (MOD(INDC-1,80) .EQ. 0) THEN
                     CALL HEADER
                     WRITE(IOUNIT,9032) CHRAVE(IAVE), (CHIDEP(II,ITYP),
     &         II=1,6),IHOUR,JDAY,IYR,GRPID(IGRP),(WORKID(K),K=1,INDGRP)
                     WRITE(IOUNIT,9044)
                     WRITE(IOUNIT,9011) CHIDEP(3,ITYP), POLLUT,
     &                                  OUTLBL(ITYP)
                     WRITE(IOUNIT,9049) CHIDEP(3,ITYP), CHIDEP(3,ITYP)
                  END IF
                  IF (MOD(INDC,2) .NE. 0) THEN
                     WRITE(BUF132(1:65),9047) SRCID(IREF(IREC)),
     &                          DIST, DIR, AVEVAL(IREC,IGRP,IAVE,ITYP)
                  ELSE
                     WRITE(BUF132(66:130),9047) SRCID(IREF(IREC)),
     &                          DIST, DIR, AVEVAL(IREC,IGRP,IAVE,ITYP)
                     WRITE(IOUNIT,9090) BUF132
                     WRITE(BUF132,9095)
                  END IF
               END IF
            END DO
            IF (MOD(INDC,2) .NE. 0) THEN
               WRITE(IOUNIT,9090) BUF132
               WRITE(BUF132,9095)
            END IF
         END IF

C        Write Out The Boundary Receptors For The Sources
         IF (IRSTAT(6) .NE. 0) THEN
            INDC = 0
            IREC = 1
            DO WHILE (IREC .LE. NUMREC)
               IF (RECTYP(IREC) .EQ. 'BD') THEN
                  INDC = INDC + 1
                  ISRF = IREF(IREC)
                  IF (MOD(INDC-1,3) .EQ. 0) THEN
                     CALL HEADER
                     WRITE(IOUNIT,9032) CHRAVE(IAVE), (CHIDEP(II,ITYP),
     &         II=1,6),IHOUR,JDAY,IYR,GRPID(IGRP),(WORKID(K),K=1,INDGRP)
                     WRITE(IOUNIT,9011) CHIDEP(3,ITYP), POLLUT,
     &                                  OUTLBL(ITYP)
                  END IF
                  WRITE(IOUNIT,9082) SRCID(ISRF), SRCTYP(ISRF),
     &      AXS(ISRF),AYS(ISRF),AZS(ISRF),CHIDEP(3,ITYP),CHIDEP(3,ITYP),
     &                CHIDEP(3,ITYP), (J, AXR(IREC+J-1), AYR(IREC+J-1),
     &                AVEVAL(IREC+J-1,IGRP,IAVE,ITYP),J=1,36)
                  IREC = IREC + 36
               ELSE
                  IREC = IREC + 1
               END IF
            END DO
         END IF

      END DO
C     End Source Group LOOP

 9011 FORMAT(/40X,'** ',A4,' OF ',A8,' IN ',A40,' **'/)
 9010 FORMAT(66(' -')/)
 9013 FORMAT(2X,F10.2,1X,'|',1X,9(F13.5))
 9016 FORMAT(3X,' Y-COORD  |',48X,'X-COORD (METERS)')
 9017 FORMAT(3X,' (METERS) |',1X,9(1X,F12.2,:))
 9018 FORMAT(3X,'DIRECTION |',48X,'DISTANCE (METERS)')
 9019 FORMAT(3X,'(DEGREES) |',1X,9(1X,F12.2,:))
 9032 FORMAT(20X,'*** CONCURRENT ',A5,1X,6A4,'VALUES',
     &       ' ENDING WITH HOUR ',I2,' FOR DAY ',I3,' OF ',I4,' ***'
     &       /24X,'FOR SOURCE GROUP:',1X,A8,
     &       /24X,'INCLUDING SOURCE(S):      ',7(A8,', ',:),
     &       /10X,12(A8,', ',:)/10X,12(A8,', ',:))
 9037 FORMAT(/35X,'*** NETWORK ID: ',A8,' ;  NETWORK TYPE: ',
     &       A8,' ***')
 9043 FORMAT(/45X,'*** DISCRETE CARTESIAN RECEPTOR POINTS ***')
 9044 FORMAT(/47X,'*** DISCRETE POLAR RECEPTOR POINTS ***')
 9045 FORMAT(6X,2(F12.2,2X),F13.5)
 9047 FORMAT(4X,A8,': ',2(F12.2,2X),F13.5)
 9048 FORMAT(6X,' X-COORD (M)   Y-COORD (M)        ',A4,
     &      22X,' X-COORD (M)   Y-COORD (M)        ',A4,/65(' -'))
 9049 FORMAT(5X,'ORIGIN',59X,'ORIGIN',
     &      /5X,' SRCID       DIST (M)     DIR (DEG)        ',A4,
     &      18X,' SRCID       DIST (M)     DIR (DEG)        ',A4,
     &      /65(' -'))
 9082 FORMAT(' BOUNDARY RECEPTOR NETWORK OF SOURCE ID: ',A8,/,5X,
     &       ' OF SOURCE TYPE: ',A8,'; WITH ORIGIN AT (',2(F10.2,', '),
     &       F10.2,')'/3(' (SEC.)  X-COORD    Y-COORD       ',A4,6X),/,
     &       12(3(1X,I4,2X,F9.1,',',F10.1,',',F13.5,' ',2X),/),/)
 9090 FORMAT(A132)
 9095 FORMAT(132(' '))

 999  RETURN
      END

      SUBROUTINE RSDUMP
C***********************************************************************
C                 RSDUMP Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: To Save Intermediate Results Arrays for Later Restart
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To incorporate modifications to date processing
C                    for Y2K compliance.  Specifically, to output the
C                    10-digit date variable (FULLDATE) with 4-digit
C                    year for date comparisons.
C                    Also modified to output arrays associated with
C                    post-1997 PM10 processing.
C                    R.W. Brode, PES, Inc., 5/12/99
C
C        MODIFIED:   Changed parameter for specifying the number of
C                    high annual/period averages from NVAL to NHIANN.
C                    R.W. Brode, PES, Inc.,  4/3/98
C
C        MODIFIED:   Changed parameter for specifying the number of
C                    high annual/period averages from NVAL to NHIANN.
C                    R.W. Brode, PES, Inc.,  4/3/98
C
C        INPUTS:  Current Date Variable
C                 Array Limits
C                 Results Arrays
C
C        OUTPUTS: Unformatted File of Intermediate Results
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, J, K, L, M

C     Variable Initializations
      MODNAM = 'RSDUMP'
      NDUMP = NDUMP + 1

C     Check for Monthly Averages and Only Dump at End of Month
      IF (MONTH .AND. .NOT.ENDMON)  GO TO 1000

      IF (SAVFIL .EQ. SAVFL2 .OR. MOD(NDUMP,2) .NE. 0) THEN
         OPEN(UNIT=IDPUNT,ERR=99,FILE=SAVFIL,FORM='UNFORMATTED',
     &        IOSTAT=IOERRN,STATUS='UNKNOWN')
         WRITE(IDPUNT) FULLDATE
         WRITE(IDPUNT) NHIVAL, NMXVAL, NUMREC, NUMGRP, NUMAVE, NUMTYP

         IF (NHIVAL .GT. 0) THEN
           WRITE(IDPUNT) (((((HIVALU(I,J,K,L,M),I=1,NUMREC),J=1,NHIVAL),
     &                       K=1,NUMGRP),L=1,NUMAVE),M=1,NUMTYP)
           WRITE(IDPUNT) (((((NHIDAT(I,J,K,L,M),I=1,NUMREC),J=1,NHIVAL),
     &                       K=1,NUMGRP),L=1,NUMAVE),M=1,NUMTYP)
           WRITE(IDPUNT) (((((HCLMSG(I,J,K,L,M),I=1,NUMREC),J=1,NHIVAL),
     &                       K=1,NUMGRP),L=1,NUMAVE),M=1,NUMTYP)

            IF (PM10AVE) THEN
               WRITE(IDPUNT) NUMYRS
               WRITE(IDPUNT) ((SUMH4H(I,J),I=1,NUMREC),J=1,NUMGRP)
            END IF

         END IF

         IF (NMXVAL .GT. 0) THEN
            WRITE(IDPUNT) ((((RMXVAL(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
            WRITE(IDPUNT) ((((MXDATE(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
            WRITE(IDPUNT) ((((MXLOCA(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
            WRITE(IDPUNT) ((((MCLMSG(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
         END IF

         IF (SEASONHR) THEN
            WRITE(IDPUNT) (((((SHVALS(I,J,K,L,M),I=1,NUMREC),
     &                J=1,NUMGRP),K=1,4),L=1,24),M=1,NUMTYP)
            WRITE(IDPUNT) ((NSEAHR(I,J),I=1,4),J=1,24)
            WRITE(IDPUNT) ((NSEACM(I,J),I=1,4),J=1,24)
         END IF

         IF (PERIOD) THEN
            WRITE(IDPUNT) IANHRS, IANCLM, IANMSG
            WRITE(IDPUNT) (((ANNVAL(I,J,K),I=1,NUMREC),J=1,NUMGRP),
     &                       K=1,NUMTYP)
            IF (MULTYR) THEN
               WRITE(IDPUNT) (((AMXVAL(I,J,K),I=1,NHIANN),J=1,NUMGRP),
     &                          K=1,NUMTYP)
               WRITE(IDPUNT) (((IMXLOC(I,J,K),I=1,NHIANN),J=1,NUMGRP),
     &                          K=1,NUMTYP)
            END IF
         ELSE IF (ANNUAL) THEN
            WRITE(IDPUNT) IANHRS, IANCLM, IANMSG, NUMYRS
            WRITE(IDPUNT) (((ANNVAL(I,J,K),I=1,NUMREC),J=1,NUMGRP),
     &                       K=1,NUMTYP)
            WRITE(IDPUNT) (((SUMANN(I,J,K),I=1,NUMREC),J=1,NUMGRP),
     &                       K=1,NUMTYP)
         END IF

         CLOSE (IDPUNT)

      ELSE
         OPEN(UNIT=IDPUN2,ERR=99,FILE=SAVFL2,FORM='UNFORMATTED',
     &        IOSTAT=IOERRN,STATUS='UNKNOWN')
         WRITE(IDPUN2) FULLDATE
         WRITE(IDPUN2) NHIVAL, NMXVAL, NUMREC, NUMGRP, NUMAVE, NUMTYP

         IF (NHIVAL .GT. 0) THEN
           WRITE(IDPUN2) (((((HIVALU(I,J,K,L,M),I=1,NUMREC),J=1,NHIVAL),
     &                       K=1,NUMGRP),L=1,NUMAVE),M=1,NUMTYP)
           WRITE(IDPUN2) (((((NHIDAT(I,J,K,L,M),I=1,NUMREC),J=1,NHIVAL),
     &                       K=1,NUMGRP),L=1,NUMAVE),M=1,NUMTYP)
           WRITE(IDPUN2) (((((HCLMSG(I,J,K,L,M),I=1,NUMREC),J=1,NHIVAL),
     &                       K=1,NUMGRP),L=1,NUMAVE),M=1,NUMTYP)

            IF (PM10AVE) THEN
               WRITE(IDPUN2) NUMYRS
               WRITE(IDPUN2) ((SUMH4H(I,J),I=1,NUMREC),J=1,NUMGRP)
            END IF

         END IF

         IF (NMXVAL .GT. 0) THEN
            WRITE(IDPUN2) ((((RMXVAL(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
            WRITE(IDPUN2) ((((MXDATE(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
            WRITE(IDPUN2) ((((MXLOCA(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
            WRITE(IDPUN2) ((((MCLMSG(I,J,K,L),I=1,NMXVAL),J=1,NUMGRP),
     &                       K=1,NUMAVE),L=1,NUMTYP)
         END IF

         IF (SEASONHR) THEN
            WRITE(IDPUN2) (((((SHVALS(I,J,K,L,M),I=1,NUMREC),
     &                J=1,NUMGRP),K=1,4),L=1,24),M=1,NUMTYP)
            WRITE(IDPUN2) ((NSEAHR(I,J),I=1,4),J=1,24)
            WRITE(IDPUN2) ((NSEACM(I,J),I=1,4),J=1,24)
         END IF

         IF (PERIOD) THEN
            WRITE(IDPUN2) IANHRS, IANCLM, IANMSG
            WRITE(IDPUN2) (((ANNVAL(I,J,K),I=1,NUMREC),J=1,NUMGRP),
     &                       K=1,NUMTYP)
            IF (MULTYR) THEN
               WRITE(IDPUN2) (((AMXVAL(I,J,K),I=1,NHIANN),J=1,NUMGRP),
     &                          K=1,NUMTYP)
               WRITE(IDPUN2) (((IMXLOC(I,J,K),I=1,NHIANN),J=1,NUMGRP),
     &                          K=1,NUMTYP)
            END IF
         ELSE IF (ANNUAL) THEN
            WRITE(IDPUN2) IANHRS, IANCLM, IANMSG, NUMYRS
            WRITE(IDPUN2) (((ANNVAL(I,J,K),I=1,NUMREC),J=1,NUMGRP),
     &                       K=1,NUMTYP)
            WRITE(IDPUN2) (((SUMANN(I,J,K),I=1,NUMREC),J=1,NUMGRP),
     &                       K=1,NUMTYP)
         END IF

         CLOSE (IDPUN2)

      END IF

      GO TO 1000

 99   CALL ERRHDL(PATH,MODNAM,'E','500','SAVEFILE')
      RUNERR = .TRUE.

 1000 RETURN
      END
