      SUBROUTINE TGCARD
C***********************************************************************
C                 TGCARD Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: To process Terrain Grid Pathway Card Images
C
C        PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    December 15, 1993
C
C        INPUTS:  Pathway (TG) and Keyword
C
C        OUTPUTS: Terrain Grid Filename
C                 Origin to use with Terrain Grid (shift in UTM coord.,
C                   must be same as that used for sources/receptors,
C                   but this is not checked!)
C                 Gridded Terrain Data
C
C        CALLED FROM:   SETUP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: IERRTG

C     Variable Initializations
      MODNAM = 'TGCARD'

      IF (KEYWRD .EQ. 'STARTING') THEN
C        Set Status Switch
         ITSTAT(1) = ITSTAT(1) + 1
         IF (ITSTAT(1) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         END IF
      ELSE IF (KEYWRD .EQ. 'INPUTFIL') THEN
C        Set Status Switch
         ITSTAT(2) = ITSTAT(2) + 1
         IF (ITSTAT(2) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Gridded Terrain File Information         ---   CALL TERFIL
            CALL TERFIL
         END IF
      ELSE IF (KEYWRD .EQ. 'LOCATION') THEN
C        Set Status Switch
         ITSTAT(3) = ITSTAT(3) + 1
         IF (ITSTAT(3) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Location (origin) Information           ---   CALL TERLOC
            CALL TERLOC
         END IF
      ELSE IF (KEYWRD .EQ. 'ELEVUNIT') THEN
C        Set Status Switch
         ITSTAT(4) = ITSTAT(4) + 1
         IF (ICSTAT(10) .NE. 0) THEN
C           Write Error Message: Use of obsolescent CO ELEVUNIT card with
C           TG ELEVUNIT card
            CALL ERRHDL(PATH,MODNAM,'E','153',' TG Path')
         ELSE IF (ITSTAT(4) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
         ELSE
C           Process Elevation Units for Source Elevations   ---   CALL TGELUN
            CALL TGELUN
         END IF
      ELSE IF (KEYWRD .EQ. 'FINISHED') THEN
C        Set Status Switch
         ITSTAT(25) = ITSTAT(25) + 1
         IF (ITSTAT(25) .NE. 1) THEN
C           WRITE Error Message: Non-repeatable Keyword
            CALL ERRHDL(PATH,MODNAM,'E','135',KEYWRD)
            GO TO 999
         END IF

C        Open Terrain Data File (Free-Format ASCII) and Process Data
         IF (ITSTAT(2) .NE. 0) THEN
            OPEN(UNIT=IZUNIT,ERR=99,FILE=TERINP,IOSTAT=IOERRN,
     &           STATUS='OLD')
            GOTO 100
C           Write Out Error Message for File OPEN Error
 99         CALL ERRHDL(PATH,MODNAM,'E','500',' TER-INP')
            GOTO 999
C                                                           ---   CALL TGDATA
100         CALL TGDATA(TGX0,TGY0,IZUNIT,TGELEV,IERRTG,
     &                  GRDXLL,GRDXUR,GRDYLL,GRDYUR,XYINT)
            IF (IERRTG .NE. 0) THEN
C              Write Out Error Message for File Grid Error
               CALL ERRHDL(PATH,MODNAM,'E','510',' TER-INP')
            ELSE
               LTGRID=.TRUE.
            END IF

C           Check to make sure that terrain grid covers all source/receptor
C           locations and check consistency of elevations   ---   CALL TGQA
            IF (LTGRID) THEN
               CALL TGQA
            END IF

         END IF

      ELSE
C        Write Error Message: Invalid Keyword for This Pathway
         CALL ERRHDL(PATH,MODNAM,'E','110',KEYWRD)
      END IF

 999  RETURN
      END

      SUBROUTINE TERFIL
C***********************************************************************
C                 TERFIL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Process Meteorology Input File Options
C                 From Runstream Input Image
C
C        PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    December 15, 1993
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Gridded Terrain Data Filename
C
C        ERROR HANDLING:   Checks for No Parameters;
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   TGCARD
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'TERFIL'

      IF (IFC .EQ. 3) THEN
C        Retrieve Data Filename as Character Substring to Maintain Case
         TERINP = RUNST1(LOCB(3):LOCE(3))
      ELSE IF (IFC .GT. 3) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Warning Message         ! No Parameters Specified
         CALL ERRHDL(PATH,MODNAM,'W','200',KEYWRD)
      END IF

      RETURN
      END

      SUBROUTINE TERLOC
C***********************************************************************
C                 TERLOC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Process Location to be used as Origin of Grid Coord.
C                 From Runstream Input Image
C
C        PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    December 15, 1993
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Origin of Cartesian System, TGX0,TGY0 (UTM m)
C
C        ERROR HANDLING:   Checks for No Parameters;
C                          Checks for No Units (uses default of m);
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   TGCARD
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'TERLOC'

      IF (IFC .EQ. 4 .OR. IFC .EQ. 5) THEN
         CALL STONUM(FIELD(3),ILEN_FLD,TGX0,IMIT)
C        Check The Numerical Field
         IF (IMIT.EQ.-1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
            GO TO 999
         END IF
         CALL STONUM(FIELD(4),ILEN_FLD,TGY0,IMIT)
C        Check The Numerical Field
         IF (IMIT.EQ.-1) THEN
            CALL ERRHDL(PATH,MODNAM,'E','208',KEYWRD)
            GO TO 999
         END IF
         IF (IFC .EQ. 5) THEN
            IF (FIELD(5) .EQ. 'FEET') THEN
               TGX0 = 0.3048 * TGX0
               TGY0 = 0.3048 * TGY0
            ELSE IF (FIELD(5) .EQ. 'KM') THEN
               TGX0 = 1000. * TGX0
               TGY0 = 1000. * TGY0
            ELSE IF (FIELD(5) .NE. 'METERS') THEN
C              WRITE Warning Message - Invalid TGUNIT Parameter
               CALL ERRHDL(PATH,MODNAM,'W','203','TGUNIT')
            END IF
         END IF
      ELSE IF (IFC .GT. 5) THEN
C        WRITE Error Message           ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message           ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200',KEYWRD)
      END IF

 999  RETURN
      END

      SUBROUTINE TGELUN
C***********************************************************************
C                 TGELUN Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Process Elevation Units Option for Terrain Grid
C                 From Runstream Input Image
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    November 22, 1994
C
C        INPUTS:  Input Runstream Image Parameters
C
C        OUTPUTS: Terrain Grid Elevation Units Switch
C
C        ERROR HANDLING:   Checks for Invalid Parameters;
C                          Checks for No Parameters;
C                          Checks for Too Many Parameters
C
C        CALLED FROM:   TGCARD
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'TGELUN'

      IF (IFC .EQ. 3) THEN
         IF (FIELD(3) .EQ. 'METERS') THEN
            TGELEV = 'METERS'
         ELSE IF (FIELD(3) .EQ. 'FEET') THEN
            TGELEV = 'FEET'
         ELSE
C           WRITE Error Message  ! Invalid Parameter
            CALL ERRHDL(PATH,MODNAM,'E','203','TG_ELEV')
         END IF
      ELSE IF (IFC .GT. 3) THEN
C        WRITE Error Message     ! Too Many Parameters
         CALL ERRHDL(PATH,MODNAM,'E','202',KEYWRD)
      ELSE
C        WRITE Error Message     ! No Parameters
         CALL ERRHDL(PATH,MODNAM,'E','200','ElevUnit')
      END IF

 999  RETURN
      END

c-----------------------------------------------------------------------
      subroutine tgdata(tgx0,tgy0,io,tgelev,ierr,grdxll,grdxur,grdyll,
     &                  grdyur,xyint)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           TGDATA
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine reads terrain elevation data (m MSL) from
c              file, and places it in array for use by ZTERR.
c
c MODIFIED:    To elevation units from feet to meters if
c              TGELEV = 'FEET'.  Roger W. Brode, PES, Inc. - 11/22/94
c
c ARGUMENTS:
c    PASSED:  tgx0      x-UTM offset of modeling coord. system (m)   [r]
c             tgy0      y-UTM offset of modeling coord. system (m)   [r]
c             io        FORTRAN unit number for gridded terrain data [i]
c             tgelev    TG elevation units option ('FEET', 'METERS') [c]
c
c  RETURNED:  ierr      error condition indicator (no error = 0)     [i]
c             grdxllm   x-coord. of lower-left corner of grid  (m)   [r]
c             grdxurm   x-coord. of upper-right corner of grid (m)   [r]
c             grdyllm   y-coord. of lower-left corner of grid  (m)   [r]
c             grdyurm   y-coord. of upper-right corner of grid (m)   [r]
c             xyint     spacing between points in grid (m)           [r]
c
c To /TGRID/
c             xllm      x-coord. of lower-left corner of grid  (m)   [r]
c             xurm      x-coord. of upper-right corner of grid (m)   [r]
c             yllm      y-coord. of lower-left corner of grid  (m)   [r]
c             yurm      y-coord. of upper-right corner of grid (m)   [r]
c             sizem     spacing between points in grid (m)           [r]
c             izarray   array of terrain data  (whole m MSL)         [i]
c
c     (NOTE:  The coordinates of the lower-left and upper-right corners
c             of the grid are returned as arguments so that they can be
c             placed in MAIN1 for QA checks against source and
c             receptor locations.)
c
c
c CALLING ROUTINES:   SETUP
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------
      USE DEPVAR
      IMPLICIT NONE

      SAVE
      INTEGER :: IO, IERR, IX, JY, IASTAT
      REAL    :: XYINT, GRDXUR, GRDXLL, GRDYUR, GRDYLL, TGX0, TGY0
      character tgelev*6, DUMMY*8, PATH*2, MODNAM*6

c --- Initialize error flag
      ierr=0
      PATH   = 'TG'
      MODNAM = 'TGDATA'

c --- Read header
      read(io,*) ntx,nty,xllm,yllm,xurm,yurm,sizem

c --- Allocate Array Storage
      ALLOCATE (IZARRAY(NTX+1,NTY+1), STAT=IASTAT)
      IF (IASTAT .NE. 0) THEN
         WRITE(DUMMY,'(I8)') IASTAT
         CALL ERRHDL(PATH,MODNAM,'E','299',DUMMY)
         IERR = 1
         GO TO 999
      END IF


c --- Reset coordinates of corners of grid to align with origin used
c --- to specify souce/receptor locations.
      xllm=xllm-tgx0
      yllm=yllm-tgy0
      xurm=xurm-tgx0
      yurm=yurm-tgy0

c --- Assign corners to variables passed back to calling subroutine
      grdxll=xllm
      grdyll=yllm
      grdxur=xurm
      grdyur=yurm
c --- Assign interval size to variable passed back to calling subroutine
      xyint = sizem

c --- Read data into array
      do jy=1,nty
         read(io,*) (izarray(ix,jy),ix=1,ntx)
      enddo

c --- Check for units conversion from feet to meters - R. Brode 11/22/94
      if (tgelev .eq. 'FEET') THEN
         do jy=1,nty
            do ix=1,ntx
               izarray(ix,jy) = izarray(ix,jy) * 0.3048
            enddo
         enddo
      endif

999   return
      end

c-----------------------------------------------------------------------
      subroutine tgqa
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           TGQA
c               D. Strimaitis, SRC
c
c PURPOSE:     Subroutine checks source/receptor locations against
c              the corners of the terrain grid to assure that all lie
c              within the grid.
c
c MODIFIED:    To use new FUNCTION ZINTERP (based on original FUNCION ZTERR)
c              to interpolate elevations for sources and receptors.
c              Roger W. Brode, PES, Inc. - 12/29/97
c
c MODIFIED:    To compare interpolated elevations from grid file against
c              source elevations and receptor elevations.
c              Roger W. Brode, PES, Inc. - 11/29/94
c
c CALLING ROUTINES:   SETUP
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

c --- Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I
      REAL    :: XSMIN, YSMIN, XSMAX, YSMAX, XRMIN, YRMIN, XRMAX, YRMAX,
     &           XLLTEST, YLLTEST, XURTEST, YURTEST, ZINT, ZINTERP, DIFF

c --- Define a test logical
      logical lfail
      data lfail/.FALSE./

c --- Variable Initializations
      modnam = 'TGQA'

c --- Loop over sources to find max/min x and y coordinates
c --- (Does NOT treat Area Sources !!)
      xsmin=axs(1)
      ysmin=ays(1)
      xsmax=axs(1)
      ysmax=ays(1)
      do i=2,numsrc
         if(axs(i) .GT. xsmax) then
            xsmax=axs(i)
         elseif(axs(i) .LT. xsmin) then
            xsmin=axs(i)
         endif
         if(ays(i) .GT. ysmax) then
            ysmax=ays(i)
         elseif(ays(i) .LT. ysmin) then
            ysmin=ays(i)
         endif
      enddo

c --- Loop over receptors to find max/min x and y coordinates
      xrmin=axr(1)
      yrmin=ayr(1)
      xrmax=axr(1)
      yrmax=ayr(1)
      do i=2,numrec
         if(axr(i) .GT. xrmax) then
            xrmax=axr(i)
         elseif(axr(i) .LT. xrmin) then
            xrmin=axr(i)
         endif
         if(ayr(i) .GT. yrmax) then
            yrmax=ayr(i)
         elseif(ayr(i) .LT. yrmin) then
            yrmin=ayr(i)
         endif
      enddo

c --- Test max/min against corners of terrain grid
      xlltest=MIN(xsmin,xrmin)
      ylltest=MIN(ysmin,yrmin)
      xurtest=MAX(xsmax,xrmax)
      yurtest=MAX(ysmax,yrmax)
      if(xlltest .LT. grdxll) lfail=.TRUE.
      if(ylltest .LT. grdyll) lfail=.TRUE.
      if(xurtest .GT. grdxur) lfail=.TRUE.
      if(yurtest .GT. grdyur) lfail=.TRUE.

      if(LFAIL) then
C        Write Error Message: Invalid Keyword for This Pathway
         call ERRHDL(PATH,MODNAM,'E','305','GRID')
         write(iounit,*) 'Lower Left of Source Range   : ',xsmin,ysmin
         write(iounit,*) 'Upper Right of Source Range  : ',xsmax,ysmax
         write(iounit,*) 'Lower Left of Receptor Range : ',xrmin,yrmin
         write(iounit,*) 'Upper Right of Receptor Range: ',xrmax,yrmax
         write(iounit,*) 'Lower Left of Terrain Grid   : ',grdxll,grdyll
         write(iounit,*) 'Upper Right of Terrain Grid  : ',grdxur,grdyur
      endif

c     Loop through sources to compare source elevations to terrain grid
      do i = 1, numsrc

c        Interpolate to obtain source elevation using FUNCTION ZINTERP
         ZINT = ZINTERP(AXS(I), AYS(I))

         diff = ABS( azs(i) - zint )
c        Compare interpolated height from terrain grid to source elevation
         if (diff .gt. 1.0 .and. diff .gt. 0.5*zint) then
            call ERRHDL(path,modnam,'W','393',srcid(i))
         end if

      enddo

c     Loop through receptors to compare receptor elevations to terrain grid
      do i = 1, numrec

c        Interpolate to obtain receptor elevation using FUNCTION ZINTERP
         ZINT = ZINTERP(AXR(I), AYR(I))

         diff = ABS( azelev(i) - zint )
c        Compare interpolated height from terrain grid to receptor elevation
         if (diff .gt. 1.0 .and. diff .gt. 0.5*zint) then
            write(dummy,'("RE#",i5.5)') i
            call ERRHDL(path,modnam,'W','394',dummy)
         end if

      enddo

      return
      end

c-----------------------------------------------------------------------
      function zinterp(xp,yp)
c-----------------------------------------------------------------------
c
c --- ISCST3    Version dated 97363 (12/29/97)
c               R.W. Brode, PES, Inc.
c               Programmed from FUNCTION ZTERR in previous versions
c               of ISCST3.  Modified to avoid array subscript out of
c               bounds at edge of grid.
c
c --- ISCST2    Version: 1.0            Level: 931215          ZTERR
c               D. Strimaitis, SRC
c
c PURPOSE:     Function computes the elevation (m MSL) at the location
c              (xp,yp), by interpolating within field of gridded terrain
c              elevations.
c
c
c ARGUMENTS:
c    PASSED:  xp        x-coordinate of the point (m)          [r]
c             yp        y-coordinate of the point (m)          [r]
c
c  RETURNED:  zinterp   value interpolated at xp,yp (m MSL)    [r]
c
c CALLING  ROUTINES:   F2INT, TGQA
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------
      USE DEPVAR
      IMPLICIT NONE

      SAVE
      INTEGER :: IXLL, IXLLP1, IYLL, IYLLP1
      REAL    :: XP, YP, ZINTERP, DI, XPOS, YPOS, TT, UU, ONEMT, ONEMU

c --- Set inverse of the size of a grid-cell
      di=1./sizem

c     - ll  denotes lower left corner of a grid-cell
c     - llm denotes lower left corner of grid-cell (1,1) -- this is the
c           lower left corner of the master terrain grid
c
c  Full development of the algorithm to obtain value at point xp,yp
c -- array index of lower left corner of cell that contains point
c     ixll=(xp-xllm)*di+1
c     iyll=(yp-yllm)*di+1
c -- position of lower left value
c     xll=xllm+sizem*(ixll-1)
c     yll=yllm+sizem*(iyll-1)
c -- fractional position of point within cell wrt lower left corner
c     tt=(xp-xll)*di
c     uu=(yp-yll)*di
c -- interpolated value
c     zi=(1.-tt)*(1.-uu)*zarray(ixll,iyll)
c    1     +tt*(1.-uu)*zarray(ixll+1,iyll)
c    2     +tt*uu*zarray(ixll+1,iyll+1)
c    3     +uu*(1.-tt)*zarray(ixll,iyll+1)

c --- Compact representation:
      xpos=(xp-xllm)*di
      ixll=INT(xpos)+1
      if (ixll .ge. ntx) ixll = ixll - 1
      tt=xpos-(ixll-1)
      onemt=1.-tt
      ixllp1=ixll+1
      ypos=(yp-yllm)*di
      iyll=INT(ypos)+1
      if (iyll .ge. nty) iyll = iyll - 1
      uu=ypos-(iyll-1)
      onemu=1.-uu
      iyllp1=iyll+1
      zinterp=onemt*onemu*FLOAT(izarray(ixll,iyll))
     1        +tt*onemu*FLOAT(izarray(ixllp1,iyll))
     2        +tt*uu*FLOAT(izarray(ixllp1,iyllp1))
     3        +uu*onemt*FLOAT(izarray(ixll,iyllp1))

      return
      end
